Attribute VB_Name = "modFinder"
'    Creatures Sprite Finder
'    Copyright (C) 2003 Vadim Trochinsky (vadim_t@teleline.es)
'
'    This program is free software; you can redistribute it and/or modify
'    it under the terms of the GNU General Public License as published by
'    the Free Software Foundation; either version 2 of the License, or
'    (at your option) any later version.
'
'    This program is distributed in the hope that it will be useful,
'    but WITHOUT ANY WARRANTY; without even the implied warranty of
'    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
'    GNU General Public License for more details.
'
'    You should have received a copy of the GNU General Public License
'    along with this program; if not, write to the Free Software
'    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


''
' Functions
'
' @author  vadim
' @version 0.1
' @date    20030716

Option Explicit

''
' Array element is a key
Const ELEM_KEY = 0

''
' Array element is a value
Const ELEM_VALUE = 1

Enum enumGameType
    C3 = 0
    C2 = 1
End Enum

''
' <b>Load a list from the registry into a ListView</b><p>
' Reads a section in the registry, retrieves all the keys in it,
' and fills a ListView, placing the values in the Text property,
' and the key names in the Tag property.
'
' @remarks This is not used for the 'Breeds' section
' @param Section Section in the registry to read
' @param List    ListView to fill
Public Sub LoadRegList(Section As String, List As ListView)
    Dim vList As Variant
    Dim i As Long, li As ListItem, elem() As String
    
    vList = GetAllSettings("CreaturesSpriteFinder", Section)
    
    For i = 0 To UBound(vList)
        Set li = List.ListItems.Add(, , vList(i, ELEM_KEY))
        On Error Resume Next
        li.SubItems(1) = vList(i, ELEM_VALUE)
        li.Tag = vList(i, ELEM_KEY)
    Next
    

End Sub

''
'<b>Make a collection with the items from a ListView</b><p>
'Clears a collection, and adds items to it, using the ~ + Tag of
'each ListItem as the key, and the Text as the text.
'
' @remarks It adds a ~ to the key in case a tag is a numeric value
' @param List ListView to get elements from
' @param Col  Collection to put the elements into
Public Sub ListToCollection(List As ListView, Col As Collection)
    Dim li As ListItem, Item As Variant
    
    For Each Item In Col
        Col.Remove 1
    Next
        
    For Each li In List.ListItems
        Col.Add li.SubItems(1), "~" & li.Tag
    Next
End Sub

''
' <b>Takes a C3 sprite filename and returns a text description of it</b><p>
' The components of the filename are returned by reference
'
' @param File   Filename to split
' @param Part   Body part
' @param Gender Species and gender
' @param Age    Age
' @param Breed  Breed
Public Sub SplitC3Filename(ByVal File As String, Part As String, Gender As String, Age As String, Breed As String)
    Part = gParts("~" & Mid(File, 1, 1))
    Gender = gGenders("~" & Mid(File, 2, 1))
    Age = gAges("~" & Mid(File, 3, 1))
    Breed = gBreedsLookup("~" & Mid(File, 4, 1) & Mid(File, 2, 1))
End Sub

''
' <b>Load a breed list from the registry into a ListView</b><p>
' Reads a section in the registry, retrieves all the keys in it,
' selects the ones that match the selected genders,
' and fills a ListView, placing the values in the Text property,
' and the key names in the Tag property.
'
' @remarks This is not used for the 'Breeds' section
' @param GenderList ListView for gender selection. The function will
'                   select all the entries in the registry that have at
'                   least one gender that matches an item in <i>GenderList</i>
'                   with .Checked = True
'
' @param List       ListView to fill
Public Sub LoadBreedList(GenderList As ListView, List As ListView)
    Dim vList As Variant, GenderKeys As String
    Dim Slot As String
    Dim i As Long, li As ListItem, elem() As String
    
    List.ListItems.Clear
    
    If GameType = C3 Then
        vList = GetAllSettings("CreaturesSpriteFinder", "Breeds")
    Else
        vList = GetAllSettings("CreaturesSpriteFinder", "C2-Breeds")
    End If
    
    ' Get a list of all the genders that are selected
    GenderKeys = SelectedItems(GenderList)
    
    For i = 0 To UBound(vList)
        ' Find if any of the genders specified in the breed match one of those we have selected
        If GenderMatches(vList(i, ELEM_KEY), GenderKeys) Then
        
            'The slot is the first character of the breed definition in the registry
            Slot = Left(vList(i, ELEM_KEY), 1)
            
            If ListItemExists(List, Slot) Then
                ' There's already an item for this slot. So we take it, and add the
                ' matching breed to it
                List.ListItems(Slot).SubItems(1) = List.ListItems(Slot).SubItems(1) & ", " & vList(i, ELEM_VALUE)
            Else
                ' No item exists, create it
                Set li = List.ListItems.Add(, Slot, Slot)
                li.SubItems(1) = vList(i, ELEM_VALUE)
                li.Tag = Slot
            End If
        End If
    Next
    
    ' Fill unknown values
    For i = Asc("A") To Asc("Z")
        If Not ListItemExists(List, Chr(i)) Then
            Set li = List.ListItems.Add(, Chr(i), Chr(i))
            li.SubItems(1) = "<unknown>"
            li.Tag = Chr(i)
        End If
    Next
End Sub

''
' <b>Load a breed list from the registry into the lookup table</b><p>
' Reads a section in the registry, retrieves all the keys in it,
' and fills the gBreedsLookup collection with the data. Missing values
' are filled with "<unknown>"
'
' @remarks Shouldn't be executed more than once
Public Sub LoadBreedsLookup()
    Dim vList As Variant, GenderKeys As String
    Dim Slot As String, Item As Variant
    Dim i As Long, li As ListItem, elem() As String
    Dim f As Long
    

    If GameType = C3 Then
        vList = GetAllSettings("CreaturesSpriteFinder", "Breeds")
    Else
        vList = GetAllSettings("CreaturesSpriteFinder", "C2-Breeds")
    End If
    
    For Each Item In gBreedsLookup
        gBreedsLookup.Remove 1
    Next
    
    
    On Error GoTo ErrH
    For i = 0 To UBound(vList)
        ' Find if any of the genders specified in the breed match one of those we have selected
        
        Slot = Left(vList(i, ELEM_KEY), 1)
        
        For f = 2 To Len(vList(i, ELEM_KEY))
            gBreedsLookup.Add vList(i, ELEM_VALUE), "~" & Slot & Mid(vList(i, ELEM_KEY), f, 1)
        Next
    Next
    
    ' Fill unknown values
    On Error Resume Next
    For i = Asc("A") To Asc("Z")
        For f = 0 To 7
            gBreedsLookup.Add "<unknown>", "~" & Chr(i) & Trim(Str(f))
        Next
    Next
    Exit Sub
    
ErrH:
    If Err.Number = 457 Then 'This key is already associated with an element of this collection
        MsgBox "Conflicting breed definition found." & vbCrLf & _
               "There seem to be two breed definitions that are trying to claim the same slot." & vbCrLf & vbCrLf & _
               "Breed '" & vList(i, ELEM_VALUE) & "' with slot " & Slot & " and genders " & Mid(vList(i, ELEM_KEY), 2) & " conflicted with" & vbCrLf & _
               "Breed '" & gBreedsLookup("~" & Slot & Mid(vList(i, ELEM_KEY), f, 1)) & "' with slot " & Slot & " and gender " & Mid(vList(i, ELEM_KEY), f, 1), vbExclamation
    Else
        MsgBox "Unexpected error #" & Err.Number & ": " & Err.Description & vbCrLf & "Please send a mail to vadim_t@teleline.es and tell me about this error", vbCritical
        End
    End If
End Sub
''
' <b>Determines whether a breed corresponds to any of the specified genders</b><p>'
'
' @param BreedKey   Breed key from the registry, in the <slot><genders> form
' @param GenderKeys Concatenated list of genders
' @return True if at least one character in GenderKeys is present in BreedKey
Public Function GenderMatches(ByVal BreedKey As String, ByVal GenderKeys As String) As Boolean
    Dim i As Long
    
    For i = 2 To Len(BreedKey)
        If InStr(GenderKeys, Mid(BreedKey, i, 1)) > 0 Then
            GenderMatches = True
            Exit Function
        End If
    Next
End Function

''
' <b>Determines whether an item with the given key exists in a ListView</b><p>
'
' @param List   ListView to search in
' @param Key    Key to look for
' @return <b>True</b> if the key is found.
Public Function ListItemExists(List As ListView, ByVal Key As String) As Boolean
    On Error GoTo ErrH
    List.ListItems(Key).Tag = List.ListItems(Key).Tag
    ListItemExists = True
    Exit Function
    
ErrH:
    'Nothing needed
End Function

Public Function CollectionItemExists(Col As Collection, ByVal Key As String) As Boolean
    On Error GoTo ErrH
    Col(Key) = Col(Key)
    CollectionItemExists = True
    Exit Function
    
ErrH:
    'Nothing needed
End Function

''
' <b>Find the files in the selected folder that match the conditions</b><p>
' Gets a list of the files in the selected folder, and then adds the ones
' that match the chosen conditions to a ListView
'
' @remarks This function is a bit slow. Perhaps it shouldn't be called automatically.
Public Sub SelectFiles()
    Dim Files As Variant, i As Long, li As ListItem
    Dim Part As String, Gender As String, Age As String, Breed As String
    Dim Pattern As String
    
    On Error Resume Next
    
    
    Files = FilesInDir(PWJoin(GameFolder, "????." & GameExt), vbNormal)

    
    If Err.Number <> 0 Then
        Status "Bad path"
        Exit Sub
    End If
    
    On Error GoTo 0
    
    Pattern = MakePattern
    frmMain.lvFiles.ListItems.Clear
    
    If UBound(Files) = 0 Then
        Status "Bad path, or no ." & GameExt & " images were found in this folder."
        Exit Sub
    End If
    
    If InStr(Pattern, "[]") > 0 Then
        Status "Incomplete selection. Please check at least one item in each list."
        Exit Sub
    End If
    
    Status "Searching for files..."
    
    For i = 0 To UBound(Files)
        If UCase(Files(i)) Like Pattern Then
            Set li = frmMain.lvFiles.ListItems.Add(, , Files(i))
            SplitC3Filename Files(i), Part, Gender, Age, Breed
            With li
                .SubItems(1) = Part
                .SubItems(2) = Gender
                .SubItems(3) = Age
                .SubItems(4) = Breed
            End With
        End If
    Next
    
    Status frmMain.lvFiles.ListItems.Count & " file(s) found."
End Sub

''
' <b>Return the concatenated Tag properties of all the checked items</b><p>
' Takes a ListView and concatenates the tags of all the checked items. This
' is used to build the Like expressions among other things
'
' @param  LV  ListView to search
' @return Concatenated tags
Public Function SelectedItems(LV As ListView) As String
    Dim li As ListItem, ret As String
        
    For Each li In LV.ListItems
        If li.Checked Then ret = ret & li.Tag
    Next
    
    SelectedItems = ret
End Function

Public Function MakePattern() As String
    With frmMain
        MakePattern = UCase(MakeSubPattern(.lvBodyParts) & _
                            MakeSubPattern(.lvGenders) & _
                            MakeSubPattern(.lvAges) & _
                            MakeSubPattern(.lvBreeds) & "." & GameExt)
    End With
End Function

Public Function MakeSubPattern(LV As ListView) As String
    
    MakeSubPattern = "[" & SelectedItems(LV) & "]"
     
End Function


Public Sub Status(Text As String)
    frmMain.StatusBar.Panels(1).Text = Text
    DoEvents
End Sub


Public Function LoadGameList() As Boolean
    Dim Paths As Variant, i As Long, li As ListItem, cur As Long
    Dim tmp As Variant
    
    Paths = GetAllSettings(APPNAME, "Paths")
    
    If VarType(Paths) = vbEmpty Then
        LoadGameList = False
        Exit Function
    End If
    
    For i = 0 To UBound(Paths)
        If Paths(i, 0) <> "Current" Then
            tmp = Split(Paths(i, 1), "|")
            frmMain.cmbGame.AddItem tmp(0)
        End If
    Next
    
    cur = GetSetting(APPNAME, "Paths", "Current", 1)
    
    If cur > frmMain.cmbGame.ListCount - 1 Or cur < 0 Then
        cur = 1
    End If
    
    frmMain.cmbGame.ListIndex = cur - 1
    LoadGameList = True
    
End Function

Public Function GameFolder() As String
    Dim Line As String, tmp As Variant
    Line = GetSetting(APPNAME, "Paths", Trim(Str(frmMain.cmbGame.ListIndex + 1)))
    tmp = Split(Line, "|")
    GameFolder = tmp(2)
End Function

Public Function GameExt() As String
    If GameType = C2 Then
        GameExt = "S16"
    ElseIf GameType = C3 Then
        GameExt = "C16"
    End If
End Function

Public Function GameType() As enumGameType
    Dim Line As String, tmp As Variant
    Line = GetSetting(APPNAME, "Paths", Trim(Str(frmMain.cmbGame.ListIndex + 1)))
    tmp = Split(Line, "|")
    
    If tmp(1) = "C2" Then
        GameType = C2
    ElseIf tmp(1) = "C3" Then
        GameType = C3
    Else
        MsgBox "Internal error: Can't recognize game type '" & tmp(1) & "'", vbCritical
        GameType = C3 'Not good, but at least will avoid crashing and let the user continue
    End If
End Function
